from typing import Annotated, Union

import strawberry

from hircine.api.types import (
    Artist,
    Character,
    Circle,
    FullArchive,
    FullComic,
    FullTag,
    Namespace,
    ScrapeComicResult,
    World,
)


@strawberry.interface
class Success:
    message: str


@strawberry.type
class AddSuccess(Success):
    id: int

    def __init__(self, modelcls, id):
        self.id = id
        self.message = f"{modelcls.__name__} added"


@strawberry.type
class AddComicSuccess(AddSuccess):
    archive_pages_remaining: bool

    def __init__(self, modelcls, id, archive_pages_remaining):
        super().__init__(modelcls, id)
        self.archive_pages_remaining = archive_pages_remaining


@strawberry.type
class UpdateSuccess(Success):
    def __init__(self):
        self.message = "Changes saved"


@strawberry.type
class UpsertSuccess(Success):
    def __init__(self):
        self.message = "Changes saved"


@strawberry.type
class DeleteSuccess(Success):
    def __init__(self):
        self.message = "Deletion successful"


@strawberry.interface
class Error:
    @strawberry.field
    def message(self) -> str:  # pragma: no cover
        return "An error occurred"


@strawberry.type
class InvalidParameterError(Error):
    parameter: str
    text: strawberry.Private[str]

    @strawberry.field
    def message(self) -> str:
        return f"Invalid parameter '{self.parameter}': {self.text}"


@strawberry.type
class IDNotFoundError(Error):
    id: int
    model: strawberry.Private[str]

    def __init__(self, modelcls, id):
        self.id = id
        self.model = modelcls.__name__

    @strawberry.field
    def message(self) -> str:
        return f"{self.model} ID not found: '{self.id}'"


@strawberry.type
class ScraperNotFoundError(Error):
    name: str

    @strawberry.field
    def message(self) -> str:
        return f"Scraper not found: '{self.name}'"


@strawberry.type
class NameExistsError(Error):
    model: strawberry.Private[str]

    def __init__(self, modelcls):
        self.model = modelcls.__name__

    @strawberry.field
    def message(self) -> str:
        return f"Another {self.model} with this name exists"


@strawberry.type
class PageClaimedError(Error):
    id: int
    comic_id: int

    @strawberry.field
    def message(self) -> str:
        return f"Page ID {self.id} is already claimed by comic ID {self.comic_id}"


@strawberry.type
class PageRemoteError(Error):
    id: int
    archive_id: int

    @strawberry.field
    def message(self) -> str:
        return f"Page ID {self.id} comes from remote archive ID {self.archive_id}"


@strawberry.type
class ScraperError(Error):
    error: str

    @strawberry.field
    def message(self) -> str:
        return f"Scraping failed: {self.error}"


@strawberry.type
class ScraperNotAvailableError(Error):
    scraper: str
    comic_id: int

    @strawberry.field
    def message(self) -> str:
        return f"Scraper {self.scraper} not available for comic ID {self.comic_id}"


@strawberry.type
class ComicTotals:
    artists: int
    characters: int
    circles: int
    tags: int
    worlds: int


@strawberry.type
class Totals:
    archives: int
    artists: int
    characters: int
    circles: int
    comics: int
    comic: ComicTotals
    images: int
    namespaces: int
    pages: int
    scrapers: int
    tags: int
    worlds: int


@strawberry.type
class Statistics:
    total: Totals


AddComicResponse = Annotated[
    Union[
        AddComicSuccess,
        IDNotFoundError,
        PageClaimedError,
        PageRemoteError,
        InvalidParameterError,
    ],
    strawberry.union("AddComicResponse"),
]
AddResponse = Annotated[
    Union[AddSuccess, IDNotFoundError, NameExistsError, InvalidParameterError],
    strawberry.union("AddResponse"),
]
ArchiveResponse = Annotated[
    Union[FullArchive, IDNotFoundError], strawberry.union("ArchiveResponse")
]
ArtistResponse = Annotated[
    Union[Artist, IDNotFoundError], strawberry.union("ArtistResponse")
]
CharacterResponse = Annotated[
    Union[Character, IDNotFoundError], strawberry.union("CharacterResponse")
]
CircleResponse = Annotated[
    Union[Circle, IDNotFoundError], strawberry.union("CircleResponse")
]
ComicResponse = Annotated[
    Union[FullComic, IDNotFoundError], strawberry.union("ComicResponse")
]
DeleteResponse = Annotated[
    Union[DeleteSuccess, IDNotFoundError], strawberry.union("DeleteResponse")
]
NamespaceResponse = Annotated[
    Union[Namespace, IDNotFoundError], strawberry.union("NamespaceResponse")
]
ScrapeComicResponse = Annotated[
    Union[
        ScrapeComicResult,
        ScraperNotFoundError,
        ScraperNotAvailableError,
        IDNotFoundError,
        ScraperError,
    ],
    strawberry.union("ScrapeComicResponse"),
]
TagResponse = Annotated[
    Union[FullTag, IDNotFoundError], strawberry.union("TagResponse")
]
UpdateResponse = Annotated[
    Union[
        UpdateSuccess,
        NameExistsError,
        IDNotFoundError,
        InvalidParameterError,
        PageRemoteError,
        PageClaimedError,
    ],
    strawberry.union("UpdateResponse"),
]
UpsertResponse = Annotated[
    Union[
        UpsertSuccess,
        NameExistsError,
        InvalidParameterError,
    ],
    strawberry.union("UpsertResponse"),
]
WorldResponse = Annotated[
    Union[World, IDNotFoundError], strawberry.union("WorldResponse")
]
