from abc import ABC, abstractmethod


class ScraperInfo:
    """
    A class containing informational data on a scraper.

    :param str name: The name of the scraper.
    :param str source: The data source, usually a well-defined name.
        For names used by built-in plugins, refer to the :ref:`Scrapers
        reference <builtin-scrapers>`.
    :param FullComic comic: The comic being scraped.
    """

    def __init__(self, name, source, comic):
        self.name = name
        self.source = source
        self.comic = comic


class ScrapeWarning(Exception):
    """
    An exception signalling a non-fatal error. Its message will be shown to the
    user once the scraping process concludes.

    This is usually raised within a callable yielded by
    :meth:`~hircine.scraper.Scraper.scrape` and should generally only be used
    to notify the user that a piece of metadata was ignored because it was
    malformed.
    """

    pass


class ScrapeError(Exception):
    """
    An exception signalling a fatal error, stopping the scraping process
    immediately.

    This should only be raised if it is impossible for the scraping process to
    continue, for example if a file or URL is inaccessible.
    """

    pass


class Scraper(ABC):
    """
    The abstract base class for scrapers.

    The following variables **must** be accessible after the instance is initialized:

    :var str name: The name of the scraper (displayed in the scraper dropdown).
    :var str source: The data source. Usually a well-defined name.
    :var bool is_available: Whether this scraper is available for the given comic.
    """

    name = "Abstract Scraper"

    source = None
    is_available = False

    def __init__(self, comic):
        """
        Initializes a scraper with the instance of the comic it is scraping.

        :param FullComic comic: The comic being scraped.
        """
        self.comic = comic
        self.warnings = []

    @abstractmethod
    def scrape(self):
        """
        A generator function that yields :ref:`scraped-data` or a callable
        returning such data.

        A callable may raise the :exc:`~hircine.scraper.ScrapeWarning`
        exception. This exception will be caught automatically and its message
        will be collected for display to the user after the scraping process concludes.
        """
        pass

    def collect(self, transformers=None):
        if transformers is None:
            transformers = []

        def generator():
            for item in self.scrape():
                if callable(item):
                    try:
                        result = item()
                    except ScrapeWarning as e:
                        self.log_warning(e)
                else:
                    result = item

                if result is not None:
                    yield result

        gen = generator()

        info = ScraperInfo(name=self.name, source=self.source, comic=self.comic)

        for fun in transformers:
            gen = fun(gen, info)

        return gen

    def log_warning(self, warning):
        self.warnings.append(warning)

    def get_warnings(self):
        return list(map(str, self.warnings))
