from dataclasses import dataclass
from datetime import date, datetime

import hircine.enums

from . import ScrapeWarning


@dataclass(frozen=True)
class Tag:
    """
    A :term:`qualified tag`, represented by strings.

    :param str namespace: The namespace.
    :param str tag: The tag.
    """

    namespace: str
    tag: str

    @classmethod
    def from_string(cls, string, delimiter=":"):
        """
        Returns a new instance of this class given a textual representation,
        usually a qualified tag in the format ``<namespace>:<tag>``. If no
        delimiter is found, the namespace is assumed to be ``none`` and the
        given string is used as a tag instead.

        :param str string: The string of text representing a qualified tag.
        :param str delimiter: The string with which the namespace is delimited
            from the tag.
        """
        match string.split(delimiter, 1):
            case [namespace, tag]:
                return cls(namespace=namespace, tag=tag)
        return cls(namespace="none", tag=string)

    def to_string(self):
        return f"{self.namespace}:{self.tag}"

    def __bool__(self):
        return bool(self.namespace) and bool(self.tag)


@dataclass(frozen=True)
class Date:
    """
    A scraped date.

    :param :class:`~datetime.date` value: The date.
    """

    value: date

    @classmethod
    def from_iso(cls, datestring):
        """
        Returns a new instance of this class given a textual representation of
        a date in the format ``YYYY-MM-DD``. See :meth:`datetime.date.fromisoformat`.

        :param str datestring: The string of text representing a date.
        :raise: :exc:`~hircine.scraper.ScrapeWarning` if the date string could
                not be parsed.
        """
        try:
            return cls(value=datetime.fromisoformat(datestring).date())
        except ValueError as e:
            raise ScrapeWarning(
                f"Could not parse date: '{datestring}' as ISO 8601"
            ) from e

    @classmethod
    def from_timestamp(cls, timestamp):
        """
        Returns a new instance of this class given a textual representation of
        a POSIX timestamp. See :meth:`datetime.date.fromtimestamp`.

        :param str timestamp: The string of text representing a POSIX timestamp.
        :raise: :exc:`~hircine.scraper.ScrapeWarning` if the timestamp could
                not be parsed.
        """
        try:
            return cls(value=datetime.fromtimestamp(int(timestamp)).date())
        except (OverflowError, OSError, ValueError) as e:
            raise ScrapeWarning(
                f"Could not parse date: '{timestamp}' as POSIX timestamp"
            ) from e

    def __bool__(self):
        return self.value is not None


@dataclass(frozen=True)
class Rating:
    """
    A scraped rating, represented by an enum.
    """

    value: hircine.enums.Rating

    def __bool__(self):
        return self.value is not None


@dataclass(frozen=True)
class Category:
    """
    A scraped category, represented by an enum.
    """

    value: hircine.enums.Category

    def __bool__(self):
        return self.value is not None


@dataclass(frozen=True)
class Censorship:
    """
    A scraped censorship specifier, represented by an enum.
    """

    value: hircine.enums.Censorship

    def __bool__(self):
        return self.value is not None


@dataclass(frozen=True)
class Language:
    """
    A scraped language, represented by an enum.
    """

    value: hircine.enums.Language

    def __bool__(self):
        return self.value is not None

    @classmethod
    def from_iso_639_3(cls, string):
        """
        Returns a new instance of this class given a case-insensitive ISO 639-3
        language code.

        :param str string: The ISO 639-3 language code.
        :raise: :exc:`~hircine.scraper.ScrapeWarning` if the language code could
                not be parsed.
        """
        try:
            return Language(value=hircine.enums.Language[string.upper()])
        except KeyError as e:
            raise ScrapeWarning(
                f"Could not parse language code: '{string}' as ISO 639-3"
            ) from e

    @classmethod
    def from_name(cls, string):
        """
        Returns a new instance of this class given a case-insensitive language name.
        Permissible language names are defined in :class:`hircine.enums.Language`.

        :param str string: The language name.
        :raise: :exc:`~hircine.scraper.ScrapeWarning` if the language name could
                not be parsed.
        """
        try:
            return Language(value=hircine.enums.Language(string.capitalize()))
        except ValueError as e:
            raise ScrapeWarning(f"Could not parse language name: '{string}'") from e


@dataclass(frozen=True)
class Direction:
    """
    A scraped direction, represented by an enum.
    """

    value: hircine.enums.Direction

    def __bool__(self):
        return self.value is not None


@dataclass(frozen=True)
class Layout:
    """
    A scraped layout, represented by an enum.
    """

    value: hircine.enums.Layout

    def __bool__(self):
        return self.value is not None


@dataclass(frozen=True)
class Title:
    """
    A scraped comic title.
    """

    value: str

    def __bool__(self):
        return bool(self.value)


@dataclass(frozen=True)
class OriginalTitle:
    """
    A scraped original title.
    """

    value: str

    def __bool__(self):
        return bool(self.value)


@dataclass(frozen=True)
class Artist:
    """
    A scraped artist.
    """

    name: str

    def __bool__(self):
        return bool(self.name)


@dataclass(frozen=True)
class Character:
    """
    A scraped character.
    """

    name: str

    def __bool__(self):
        return bool(self.name)


@dataclass(frozen=True)
class Circle:
    """
    A scraped circle.
    """

    name: str

    def __bool__(self):
        return bool(self.name)


@dataclass(frozen=True)
class World:
    """
    A scraped world.
    """

    name: str

    def __bool__(self):
        return bool(self.name)


@dataclass(frozen=True)
class URL:
    """
    A scraped URL.
    """

    value: str

    def __bool__(self):
        return bool(self.value)
