import asyncio
import os

import strawberry
import uvicorn
from starlette.applications import Starlette
from starlette.middleware import Middleware
from starlette.middleware.cors import CORSMiddleware
from starlette.routing import Mount, Route
from starlette.staticfiles import StaticFiles
from strawberry.asgi import GraphQL

import hircine.db as db
from hircine.api.mutation import Mutation
from hircine.api.query import Query
from hircine.config import init_dir_structure

schema = strawberry.Schema(query=Query, mutation=Mutation)
graphql: GraphQL = GraphQL(schema)


class SinglePageApplication(StaticFiles):  # pragma: no cover
    def __init__(self, index="index.html"):
        self.index = index
        super().__init__(
            packages=[("hircine", "static/app")], html=True, check_dir=True
        )

    def lookup_path(self, path):
        full_path, stat_result = super().lookup_path(path)

        if stat_result is None:
            return super().lookup_path(self.index)

        return (full_path, stat_result)


class HelpFiles(StaticFiles):  # pragma: no cover
    def __init__(self, index="index.html"):
        self.index = index
        super().__init__(
            packages=[("hircine", "static/help")], html=True, check_dir=True
        )


def app():  # pragma: no cover
    dirs = init_dir_structure()
    db.configure(dirs)

    routes = [
        Route("/graphql", endpoint=graphql),
        Mount("/objects", app=StaticFiles(directory=dirs.objects), name="objects"),
        Mount("/help", app=HelpFiles(), name="help"),
        Mount("/", app=SinglePageApplication(), name="app"),
    ]

    middleware = []

    if "HIRCINE_DEV" in os.environ:
        middleware = [
            Middleware(
                CORSMiddleware,
                allow_origins=["*"],
                allow_methods=["*"],
                allow_headers=["*"],
            )
        ]

    return Starlette(routes=routes, middleware=middleware)


if __name__ == "__main__":
    dirs = init_dir_structure()
    db.ensuredb(dirs)

    engine = db.configure(dirs)
    asyncio.run(db.ensure_current_revision(engine))

    uvicorn.run("hircine.app:app", host="::", reload=True, factory=True, lifespan="on")
