import hircine.enums as enums
from hircine.scraper.types import (
    URL,
    Artist,
    Category,
    Censorship,
    Character,
    Date,
    Language,
    Rating,
    Tag,
    Title,
    World,
)
from hircine.scraper.utils import parse_dict


def replace_underscore(fun):
    return lambda input: fun(input.replace("_", " "))


class E621Handler:
    source = "e621"

    ratings = {
        "e": Rating(enums.Rating.EXPLICIT),
        "q": Rating(enums.Rating.QUESTIONABLE),
        "s": Rating(enums.Rating.SAFE),
    }

    def scrape(self, data):
        match data.get("subcategory"):
            case "pool":
                yield from self.scrape_pool(data)

    def scrape_pool(self, data):
        parsers = {
            "date": Date.from_iso,
            "rating": self.ratings.get,
            "pool": {
                "id": lambda pid: URL(f"https://e621.net/pools/{pid}"),
                "name": Title,
            },
            "tags": {
                "general": replace_underscore(Tag.from_string),
                "artist": replace_underscore(Artist),
                "character": replace_underscore(Character),
                "copyright": replace_underscore(World),
                "species": replace_underscore(Tag.from_string),
                "meta": self.parse_meta,
            },
        }

        self.is_likely_uncensored = True

        yield from parse_dict(parsers, data)

        if self.is_likely_uncensored:
            yield Censorship(enums.Censorship.NONE)

    def parse_meta(self, input):
        match input:
            case "comic":
                return Category(enums.Category.COMIC)
            case "censor_bar":
                self.is_likely_uncensored = False
                return Censorship(enums.Censorship.BAR)
            case "mosaic_censorship":
                self.is_likely_uncensored = False
                return Censorship(enums.Censorship.MOSAIC)
            case "uncensored":
                return Censorship(enums.Censorship.NONE)

        if input.endswith("_text"):
            lang, _ = input.split("_text", 1)

            return Language.from_name(lang)
