import os
from datetime import UTC, date, datetime

from sqlalchemy import (
    DateTime,
    ForeignKey,
    MetaData,
    TypeDecorator,
    event,
    func,
    select,
)
from sqlalchemy.orm import (
    DeclarativeBase,
    Mapped,
    declared_attr,
    deferred,
    joinedload,
    mapped_column,
    relationship,
    selectinload,
)

from hircine.api import APIException
from hircine.api.responses import InvalidParameterError
from hircine.enums import Category, Censorship, Direction, Language, Layout, Rating

naming_convention = {
    "ix": "ix_%(column_0_label)s",
    "ck": "ck_%(table_name)s_%(constraint_name)s",
}


class DateTimeUTC(TypeDecorator):
    impl = DateTime
    cache_ok = True

    def process_bind_param(self, value, dialect):
        if value is not None:
            if not value.tzinfo:
                raise TypeError("tzinfo is required")
            value = value.astimezone(UTC).replace(tzinfo=None)
        return value

    def process_result_value(self, value, dialect):
        if value is not None:
            value = value.replace(tzinfo=UTC)
        return value


class Base(DeclarativeBase):
    metadata = MetaData(naming_convention=naming_convention)

    @declared_attr.directive
    def __tablename__(cls) -> str:
        return cls.__name__.lower()

    __mapper_args__ = {"eager_defaults": True}

    @classmethod
    def load_update(cls, fields):
        return []


class MixinID:
    id: Mapped[int] = mapped_column(primary_key=True)


class MixinName:
    name: Mapped[str] = mapped_column(unique=True)

    @classmethod
    def default_order(cls):
        return [cls.name]


class MixinFavourite:
    favourite: Mapped[bool] = mapped_column(insert_default=False)


class MixinOrganized:
    organized: Mapped[bool] = mapped_column(insert_default=False)


class MixinBookmarked:
    bookmarked: Mapped[bool] = mapped_column(insert_default=False)


class MixinCreatedAt:
    created_at: Mapped[datetime] = mapped_column(DateTimeUTC, server_default=func.now())


class MixinModifyDates(MixinCreatedAt):
    updated_at: Mapped[datetime] = mapped_column(DateTimeUTC, server_default=func.now())


class Archive(MixinID, MixinCreatedAt, MixinOrganized, Base):
    hash: Mapped[str] = mapped_column(unique=True)
    path: Mapped[str] = mapped_column(unique=True)
    size: Mapped[int]
    mtime: Mapped[datetime] = mapped_column(DateTimeUTC)

    cover_id: Mapped[int] = mapped_column(ForeignKey("image.id"))
    cover: Mapped["Image"] = relationship(lazy="joined", innerjoin=True)

    pages: Mapped[list["Page"]] = relationship(
        back_populates="archive",
        order_by="(Page.index)",
        cascade="save-update, merge, expunge, delete, delete-orphan",
    )
    comics: Mapped[list["Comic"]] = relationship(
        back_populates="archive",
        cascade="save-update, merge, expunge, delete, delete-orphan",
    )

    page_count: Mapped[int]

    @property
    def name(self):
        return os.path.basename(self.path)

    @classmethod
    def default_order(cls):
        return [cls.path]

    @classmethod
    def load_full(cls):
        return [
            joinedload(cls.pages, innerjoin=True),
            selectinload(cls.comics),
        ]


class Image(MixinID, Base):
    hash: Mapped[str] = mapped_column(unique=True)
    width: Mapped[int]
    height: Mapped[int]

    @property
    def aspect_ratio(self):
        return self.width / self.height


class Page(MixinID, Base):
    path: Mapped[str]
    index: Mapped[int]

    archive_id: Mapped[int] = mapped_column(ForeignKey("archive.id"))
    archive: Mapped["Archive"] = relationship(back_populates="pages")

    image_id: Mapped[int] = mapped_column(ForeignKey("image.id"))
    image: Mapped["Image"] = relationship(lazy="joined", innerjoin=True)

    comic_id: Mapped[int | None] = mapped_column(ForeignKey("comic.id"))


class Comic(
    MixinID, MixinModifyDates, MixinFavourite, MixinOrganized, MixinBookmarked, Base
):
    title: Mapped[str]
    original_title: Mapped[str | None]
    url: Mapped[str | None]
    language: Mapped[Language | None]
    date: Mapped[date | None]

    direction: Mapped[Direction] = mapped_column(insert_default=Direction.LEFT_TO_RIGHT)
    layout: Mapped[Layout] = mapped_column(insert_default=Layout.SINGLE)
    rating: Mapped[Rating | None]
    category: Mapped[Category | None]
    censorship: Mapped[Censorship | None]

    cover_id: Mapped[int] = mapped_column(ForeignKey("image.id"))
    cover: Mapped["Image"] = relationship(lazy="joined", innerjoin=True)

    archive_id: Mapped[int] = mapped_column(ForeignKey("archive.id"))
    archive: Mapped["Archive"] = relationship(back_populates="comics")

    pages: Mapped[list["Page"]] = relationship(order_by="(Page.index)")
    page_count: Mapped[int]

    tags: Mapped[list["ComicTag"]] = relationship(
        lazy="selectin",
        cascade="save-update, merge, expunge, delete, delete-orphan",
        passive_deletes=True,
    )

    artists: Mapped[list["Artist"]] = relationship(
        secondary="comicartist",
        lazy="selectin",
        order_by="(Artist.name, Artist.id)",
        passive_deletes=True,
    )

    characters: Mapped[list["Character"]] = relationship(
        secondary="comiccharacter",
        lazy="selectin",
        order_by="(Character.name, Character.id)",
        passive_deletes=True,
    )

    circles: Mapped[list["Circle"]] = relationship(
        secondary="comiccircle",
        lazy="selectin",
        order_by="(Circle.name, Circle.id)",
        passive_deletes=True,
    )

    worlds: Mapped[list["World"]] = relationship(
        secondary="comicworld",
        lazy="selectin",
        order_by="(World.name, World.id)",
        passive_deletes=True,
    )

    @classmethod
    def default_order(cls):
        return [cls.title]

    @classmethod
    def load_full(cls):
        return [
            joinedload(cls.archive, innerjoin=True),
            joinedload(cls.pages, innerjoin=True),
        ]

    @classmethod
    def load_update(cls, fields):
        if "pages" in fields:
            return [joinedload(cls.pages, innerjoin=True)]
        return []


class Tag(MixinID, MixinModifyDates, MixinName, Base):
    description: Mapped[str | None]
    namespaces: Mapped[list["Namespace"]] = relationship(
        secondary="tagnamespaces",
        passive_deletes=True,
        order_by="(Namespace.sort_name, Namespace.name, Namespace.id)",
    )

    @classmethod
    def load_full(cls):
        return [selectinload(cls.namespaces)]

    @classmethod
    def load_update(cls, fields):
        if "namespaces" in fields:
            return cls.load_full()
        return []


class Namespace(MixinID, MixinModifyDates, MixinName, Base):
    sort_name: Mapped[str | None]

    @classmethod
    def default_order(cls):
        return [cls.sort_name, cls.name]

    @classmethod
    def load_full(cls):
        return []


class TagNamespaces(Base):
    namespace_id: Mapped[int] = mapped_column(
        ForeignKey("namespace.id", ondelete="CASCADE"), primary_key=True
    )
    tag_id: Mapped[int] = mapped_column(
        ForeignKey("tag.id", ondelete="CASCADE"), primary_key=True
    )


class ComicTag(Base):
    comic_id: Mapped[int] = mapped_column(
        ForeignKey("comic.id", ondelete="CASCADE"), primary_key=True
    )
    namespace_id: Mapped[int] = mapped_column(
        ForeignKey("namespace.id", ondelete="CASCADE"), primary_key=True
    )
    tag_id: Mapped[int] = mapped_column(
        ForeignKey("tag.id", ondelete="CASCADE"), primary_key=True
    )

    namespace: Mapped["Namespace"] = relationship(
        lazy="joined",
        innerjoin=True,
        order_by="(Namespace.sort_name, Namespace.name, Namespace.id)",
    )

    tag: Mapped["Tag"] = relationship(
        lazy="joined",
        innerjoin=True,
        order_by="(Tag.name, Tag.id)",
    )

    @property
    def name(self):
        return f"{self.namespace.name}:{self.tag.name}"

    @property
    def id(self):
        return f"{self.namespace.id}:{self.tag.id}"


class Artist(MixinID, MixinModifyDates, MixinName, Base):
    pass


class ComicArtist(Base):
    comic_id: Mapped[int] = mapped_column(
        ForeignKey("comic.id", ondelete="CASCADE"), primary_key=True
    )
    artist_id: Mapped[int] = mapped_column(
        ForeignKey("artist.id", ondelete="CASCADE"), primary_key=True
    )


class Character(MixinID, MixinModifyDates, MixinName, Base):
    pass


class ComicCharacter(Base):
    comic_id: Mapped[int] = mapped_column(
        ForeignKey("comic.id", ondelete="CASCADE"), primary_key=True
    )
    character_id: Mapped[int] = mapped_column(
        ForeignKey("character.id", ondelete="CASCADE"), primary_key=True
    )


class Circle(MixinID, MixinModifyDates, MixinName, Base):
    pass


class ComicCircle(Base):
    comic_id: Mapped[int] = mapped_column(
        ForeignKey("comic.id", ondelete="CASCADE"), primary_key=True
    )
    circle_id: Mapped[int] = mapped_column(
        ForeignKey("circle.id", ondelete="CASCADE"), primary_key=True
    )


class World(MixinID, MixinModifyDates, MixinName, Base):
    pass


class ComicWorld(Base):
    comic_id: Mapped[int] = mapped_column(
        ForeignKey("comic.id", ondelete="CASCADE"), primary_key=True
    )
    world_id: Mapped[int] = mapped_column(
        ForeignKey("world.id", ondelete="CASCADE"), primary_key=True
    )


def defer_relationship_count(relationship, secondary=False):
    if secondary:
        left, right = relationship.property.secondary_synchronize_pairs[0]
    else:
        left, right = relationship.property.synchronize_pairs[0]

    return deferred(
        select(func.count(right))
        .select_from(right.table)
        .where(left == right)
        .scalar_subquery()
    )


Comic.artist_count = defer_relationship_count(Comic.artists)
Comic.character_count = defer_relationship_count(Comic.characters)
Comic.circle_count = defer_relationship_count(Comic.circles)
Comic.tag_count = defer_relationship_count(Comic.tags)
Comic.world_count = defer_relationship_count(Comic.worlds)

Artist.comic_count = defer_relationship_count(Comic.artists, secondary=True)
Character.comic_count = defer_relationship_count(Comic.characters, secondary=True)
Circle.comic_count = defer_relationship_count(Comic.circles, secondary=True)
Namespace.tag_count = defer_relationship_count(Tag.namespaces, secondary=True)
Tag.comic_count = deferred(
    select(func.count(ComicTag.tag_id))
    .where(Tag.id == ComicTag.tag_id)
    .scalar_subquery()
)
Tag.namespace_count = defer_relationship_count(Tag.namespaces)
World.comic_count = defer_relationship_count(Comic.worlds, secondary=True)


@event.listens_for(Comic.pages, "bulk_replace")
def on_comic_pages_bulk_replace(target, values, initiator):
    if not values:
        raise APIException(
            InvalidParameterError(parameter="pages", text="cannot be empty")
        )

    target.page_count = len(values)
