import re

import yaml

import hircine.enums as enums
from hircine.scraper import Scraper
from hircine.scraper.types import (
    URL,
    Artist,
    Censorship,
    Circle,
    Date,
    Direction,
    Language,
    Rating,
    Tag,
    Title,
    World,
)
from hircine.scraper.utils import open_archive_file, parse_dict

ANCHIRA_REGEX = re.compile(r"^https?://anchira\.to/g/")
NEXUS_REGEX = re.compile(r"^https?://hentainexus\.com/")


class AnchiraYamlScraper(Scraper):
    """
    A scraper for ``info.yaml`` files found in archives downloaded from
    *anchira.to*.

    .. list-table::
       :align: left

       * - **Requires**
         - ``info.yaml`` in the archive or as a sidecar.
       * - **Source**
         - ``anchira.to``
    """

    name = "anchira.to info.yaml"
    source = "anchira.to"

    def __init__(self, comic):
        super().__init__(comic)

        self.data = self.load()
        source = self.data.get("Source")

        if source:
            if re.match(ANCHIRA_REGEX, source) or re.match(NEXUS_REGEX, source):
                self.is_available = True
        else:
            # heuristic, but should be good enough
            url = self.data.get("URL")
            parody = self.data.get("Parody")

            self.is_available = url is not None and parody is not None

    def load(self):
        try:
            with open_archive_file(self.comic.archive, "info.yaml") as yif:
                return yaml.safe_load(yif)
        except Exception:
            return {}

    def scrape(self):
        parsers = {
            "Title": Title,
            "Artist": Artist,
            "URL": URL,
            "Released": Date.from_timestamp,
            "Circle": Circle,
            "Parody": self.parse_world,
            "Tags": self.parse_tag,
        }

        yield from parse_dict(parsers, self.data)

        yield Language(enums.Language.EN)
        yield Direction(enums.Direction.RIGHT_TO_LEFT)

    def parse_world(self, input):
        match input:
            case "Original Work":
                return

        return World(input)

    def parse_tag(self, input):
        match input:
            case "Unlimited":
                return
            case "Hentai":
                return Rating(value=enums.Rating.EXPLICIT)
            case "Non-H" | "Ecchi":
                return Rating(value=enums.Rating.QUESTIONABLE)
            case "Uncensored":
                return Censorship(value=enums.Censorship.NONE)
            case _:
                return Tag.from_string(input)
