import html
import json
import re

import requests

from hircine.scraper import ScrapeError, Scraper

from .handlers.exhentai import ExHentaiHandler

API_URL = "https://api.e-hentai.org/api.php"
URL_REGEX = re.compile(
    r"^https?://(?:exhentai|e-hentai).org/g/(?P<id>\d+)/(?P<token>[0-9a-fA-F]+).*"
)


class EHentaiAPIScraper(Scraper):
    """
    A scraper for the `E-Hentai API <https://ehwiki.org/wiki/API>`_.

    .. list-table::
       :align: left

       * - **Requires**
         - The comic :attr:`URL <hircine.api.types.FullComic.url>` pointing to
           a gallery on *e-hentai.org* or *exhentai.org*
       * - **Source**
         - ``exhentai``

    """

    name = "e-hentai.org API"
    source = "exhentai"

    def __init__(self, comic):
        super().__init__(comic)

        if self.comic.url:
            match = re.fullmatch(URL_REGEX, self.comic.url)

            if match:
                self.is_available = True
                self.id = int(match.group("id"))
                self.token = match.group("token")

    def scrape(self):
        data = json.dumps(
            {
                "method": "gdata",
                "gidlist": [[self.id, self.token]],
                "namespace": 1,
            },
            separators=(",", ":"),
        )

        request = requests.post(API_URL, data=data)

        if request.status_code != requests.codes.ok:
            raise ScrapeError(f"Request failed with status code {request.status_code}")

        try:
            response = json.loads(request.text)["gmetadata"][0]
        except json.JSONDecodeError as err:
            raise ScrapeError("Could not parse JSON response") from err
        except (KeyError, IndexError) as err:
            raise ScrapeError("Response is missing 'gmetadata' field") from err

        title = response.get("title")
        if title:
            response["title"] = html.unescape(title)

        title_jpn = response.get("title_jpn")
        if title_jpn:
            response["title_jpn"] = html.unescape(title_jpn)

        handler = ExHentaiHandler()
        yield from handler.scrape(response)
