import json

from hircine.scraper import Scraper
from hircine.scraper.utils import open_archive_file

from .handlers.dynastyscans import DynastyScansHandler
from .handlers.e621 import E621Handler
from .handlers.exhentai import ExHentaiHandler
from .handlers.mangadex import MangadexHandler

HANDLERS = {
    "dynastyscans": DynastyScansHandler,
    "e621": E621Handler,
    "exhentai": ExHentaiHandler,
    "mangadex": MangadexHandler,
}


class GalleryDLScraper(Scraper):
    """
    A scraper for `gallery-dl's <https://github.com/mikf/gallery-dl>`_
    ``info.json`` files. For now supports only a select subset of extractors.

    .. list-table::
       :align: left

       * - **Requires**
         - ``info.json`` in the archive or as a sidecar.
       * - **Source**
         - ``dynastyscans``, ``e621``, ``exhentai``, ``mangadex``
    """

    def __init__(self, comic):
        super().__init__(comic)

        self.data = self.load()
        category = self.data.get("category")

        if category in HANDLERS:
            self.is_available = True

            self.handler = HANDLERS.get(category)()
            self.source = self.handler.source
            self.name = f"gallery-dl info.json ({self.source})"

    def load(self):
        try:
            with open_archive_file(self.comic.archive, "info.json") as jif:
                return json.load(jif)
        except Exception:
            return {}

    def scrape(self):
        yield from self.handler.scrape(self.data)
