import re

import yaml

import hircine.enums as enums
from hircine.scraper import Scraper
from hircine.scraper.types import (
    Artist,
    Censorship,
    Circle,
    Direction,
    Language,
    Tag,
    Title,
)
from hircine.scraper.utils import open_archive_file, parse_dict

SOURCE_REGEX = re.compile(r"^SchaleNetwork:")


class SchaleNetworkScraper(Scraper):
    """
    A scraper for ``info.yaml`` files found in archives downloaded from
    *schale.network*.

    .. list-table::
       :align: left

       * - **Requires**
         - ``info.yaml`` in the archive or as a sidecar.
       * - **Source**
         - ``schale.network``
    """

    name = "schale.network info.yaml"
    source = "schale.network"

    def __init__(self, comic):
        super().__init__(comic)

        self.data = self.load()
        source = self.data.get("source")

        if source and re.match(SOURCE_REGEX, source):
            self.is_available = True

    def load(self):
        try:
            with open_archive_file(self.comic.archive, "info.yaml") as yif:
                return yaml.safe_load(yif)
        except Exception:
            return {}

    def scrape(self):
        parsers = {
            "title": Title,
            "artist": Artist,
            "circle": Circle,
            "general": Tag.from_string,
            "male": lambda s: Tag(namespace="male", tag=s),
            "female": lambda s: Tag(namespace="female", tag=s),
            "mixed": lambda s: Tag(namespace="mixed", tag=s),
            "language": self.parse_language,
            "other": self.parse_other,
        }

        yield from parse_dict(parsers, self.data)

        yield Direction(enums.Direction.RIGHT_TO_LEFT)

    def parse_language(self, input):
        if not input or input in ["translated"]:
            return

        return Language.from_name(input)

    def parse_other(self, input):
        match input:
            case "uncensored":
                return Censorship(value=enums.Censorship.NONE)
            case _:
                return Tag.from_string(input)
